/**
 * \file sdc_op_common.h
 *
 * \brief libSDC: Definition of functions etc. required by convenience and
 * non convenience operation API - <b>should not be included directly!</b>
 *
 * Application should not include this header directly!
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \ingroup libSDC_API
 * \defgroup operations Cryptographic operations
 * Cryptographic operations
 */

/**
 * \ingroup operations
 * \defgroup operations_encdec Encrypt/Decrypt
 * API to perform encryption and decryption without authentication
 */
/**
 * \ingroup operations_encdec
 * \defgroup operations_encdec_common Common
 * Common functions (e.g. to get the default setting for algorithm and block mode)
 */
/**
 * \ingroup operations
 * \defgroup operations_signverify Sign/Verify
 * API to perform authentication
 */
/**
 * \ingroup operations_signverify
 * \defgroup operations_signverify_common Common
 * Common functions (e.g. to get the default setting for algorithm and block mode)
 */
/**
 * \ingroup operations
 * \defgroup operations_digest Digest
 * API to perform authentication
 */
/**
 * \ingroup operations_digest
 * \defgroup operations_digest_common Common
 * Common functions (e.g. to get the default setting for hash mode)
 */
/**
 * \ingroup operations
 * \defgroup operations_wrapunwrap Wrap/Unwrap
 * API to perform encryption and decryption with authentication
 */
/**
 * \ingroup operations_wrapunwrap
 * \defgroup operations_wrapunwrap_common Common
 * Common functions (e.g. to get the default setting for algorithm and block mode)
 */
/**
 * \ingroup operations
 * \defgroup operations_common Common
 * Common functions/defines
 */


#ifndef __SDC_LIB_OP_COMMON_H_
#define __SDC_LIB_OP_COMMON_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>

/* Definitions types and defaults */

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Define to select architecture and type dependent default IV length
 */
#define SDC_IV_USE_DEFAULT SIZE_MAX

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Define to select architecture and type dependent default tag (mac or signature) length
 */
#define SDC_TAG_USE_DEFAULT SIZE_MAX

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Deprecated : SDC_SIG_USE_DEFAULT is now named SDC_TAG_USE_DEFAULT
 */
#define SDC_SIG_USE_DEFAULT    SDC_TAG_USE_DEFAULT

/**
 * \ingroup operations_digest_common
 * \brief Define to select architecture and type dependent default digest length
 */
#define SDC_DGST_USE_DEFAULT SIZE_MAX

/**
 * \ingroup operations
 * \brief Define to specify that no special options shall be selected via
 * \ref sdc_wrap_unwrap_type_set_opt_bmsk, \ref sdc_encrypt_decrypt_type_set_opt_bmsk,
 * \ref sdc_sign_verify_type_set_opt_bmsk or \ref sdc_dgst_type_set_opt_bmsk
 */
#define SDC_NO_OPT_BMSK 0

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Specification of the algorithm and block mode to be used for wrap or unwrap
 */
typedef struct sdc_wrap_unwrap_type sdc_wrap_unwrap_type_t;

/**
 * \ingroup operations_wrapunwrap_common
 * \brief libSDC algorithms available for wrap/unwrap
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all algorithms need to be available for each architecture!</b>
 */
typedef enum {
    SDC_WRAP_ALG_INVALID,
    SDC_WRAP_ALG_FIRST,                     /**< helper to iterate over algorithms */
    SDC_WRAP_ALG_AES = SDC_WRAP_ALG_FIRST,  /**< Algorithm AES */
    SDC_WRAP_ALG_END                        /**< helper to iterate over algorithms */
} sdc_wrap_unwrap_alg_t;

/**
 * \ingroup operations_wrapunwrap_common
 * \brief libSDC block modes available for wrap/unwrap
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all block modes need to be available for each architecture!</b>
 */
typedef enum {
    SDC_WRAP_BLK_INVALID,
    SDC_WRAP_BLK_FIRST,                     /**< helper to iterate over block modes */
    SDC_WRAP_BLK_CCM = SDC_WRAP_BLK_FIRST,  /**< Block mode CCM */
    SDC_WRAP_BLK_GCM,                       /**< Block mode GCM */
    SDC_WRAP_BLK_END                        /**< helper to iterate over block modes */
} sdc_wrap_unwrap_blk_t;


/**
 * \ingroup operations_encdec_common
 * \brief Specification of the algorithm and block mode to be used for encrypt or decrypt
 */
typedef struct sdc_encrypt_decrypt_type sdc_encrypt_decrypt_type_t;

/**
 * \ingroup operations_encdec_common
 * \brief libSDC algorithms available for encrypt/decrypt
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all algorithms need to be available for each architecture!</b>
 */
typedef enum {
    SDC_ENCDEC_ALG_INVALID,
    SDC_ENCDEC_ALG_FIRST,                     /**< helper to iterate over algorithms */
    SDC_ENCDEC_ALG_AES = SDC_ENCDEC_ALG_FIRST,  /**< Algorithm AES */
    SDC_ENCDEC_ALG_DES,                       /**< Algorithm DES */
    SDC_ENCDEC_ALG_3DES,                      /**< Algorithm 3DES */
    SDC_ENCDEC_ALG_RSA,                       /**< Algorithm RSA */
    SDC_ENCDEC_ALG_END                        /**< helper to iterate over algorithms */
} sdc_encrypt_decrypt_alg_t;

/**
 * \ingroup operations_encdec_common
 * \brief libSDC block modes available for wrap/unwrap
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all block modes need to be available for each architecture!</b>
 */
typedef enum {
    SDC_ENCDEC_BLK_INVALID,
    SDC_ENCDEC_BLK_FIRST,                     /**< helper to iterate over block modes */
    SDC_ENCDEC_BLK_ECB = SDC_ENCDEC_BLK_FIRST,  /**< Block mode ECB */
    SDC_ENCDEC_BLK_CTR,                       /**< Block mode CTR */
    SDC_ENCDEC_BLK_CBC,                       /**< Block mode CBC */
    SDC_ENCDEC_BLK_CFB,                       /**< Block mode CFB */
    SDC_ENCDEC_BLK_OFB,                       /**< Block mode OFB */
    SDC_ENCDEC_BLK_NONE,                      /**< No Block mode */
    SDC_ENCDEC_BLK_END                        /**< helper to iterate over block modes */
} sdc_encrypt_decrypt_blk_t;

/**
 * \ingroup operations_signverify_common
 * \brief Specification of the algorithm and block mode to be used for sign or verify
 */
typedef struct sdc_sign_verify_type sdc_sign_verify_type_t;

/**
 * \ingroup operations_signverify_common
 * \brief libSDC algorithms available for sign/verify
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all algorithms need to be available for each architecture!</b>
 */
typedef enum {
    SDC_SIGNVER_ALG_INVALID,
    SDC_SIGNVER_ALG_FIRST,                     /**< helper to iterate over algorithms */
    SDC_SIGNVER_ALG_HMAC = SDC_SIGNVER_ALG_FIRST,  /**< HMAC */
    SDC_SIGNVER_ALG_RSA,
    SDC_SIGNVER_ALG_END                        /**< helper to iterate over algorithms */
} sdc_sign_verify_alg_t;

/**
 * \ingroup operations_signverify_common
 * \brief libSDC hash algorithms available for sign/verify
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all block modes need to be available for each architecture!</b>
 */
typedef enum {
    SDC_SIGNVER_HASH_INVALID,
    SDC_SIGNVER_HASH_FIRST,               /**< helper to iterate over hash modes */
    SDC_SIGNVER_HASH_MD5 = SDC_SIGNVER_HASH_FIRST,  /**< hash MD5 */
    SDC_SIGNVER_HASH_SHA1,                /**< hash SHA1 */
    SDC_SIGNVER_HASH_SHA224,              /**< hash SHA224 */
    SDC_SIGNVER_HASH_SHA256,              /**< hash SHA256 */
    SDC_SIGNVER_HASH_SHA384,              /**< hash SHA384 */
    SDC_SIGNVER_HASH_SHA512,              /**< hash SHA512 */
    SDC_SIGNVER_HASH_END                  /**< helper to iterate over hash modes */
} sdc_sign_verify_hash_t;

/**
 * \ingroup operations_signverify_common
 * \brief Option bmsk value to select that pre-computed hash is provided instead of normal data
 *
 * Normally the sign/verify operations will sign/verify an internally computed hash.
 * When this bmsk is set and the \ref sdc_sign_verify_type_t supports it, the hash is provided
 * instead of data and directly used for sign/verify.
 */
#define SDC_SIGNVER_OPT_PRECOMP_HASH 0x00000002

/**
 * \ingroup operations_signverify_common
 * \brief Helper to easily check if bits in opt_bmsk of \ref sdc_sign_verify_type_set_opt_bmsk not belonging to any option are set
 */
#define SDC_SIGNVER_OPTS_ALL (SDC_SIGNVER_OPT_PRECOMP_HASH)


/**
 * \ingroup operations_digest_common
 * \brief Specification of the hash algorithm for digest
 */
typedef struct sdc_dgst_type sdc_dgst_type_t;

/**
 * \ingroup operations_digest_common
 * \brief libSDC hash algorithms available for dgst
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all hash modes need to be available for each architecture!</b>
 */
typedef enum {
    SDC_DGST_HASH_INVALID,
    SDC_DGST_HASH_FIRST,               /**< helper to iterate over hash modes */
    SDC_DGST_HASH_MD5 = SDC_DGST_HASH_FIRST,  /**< DGST hash MD5 */
    SDC_DGST_HASH_SHA1,                /**< DGST hash SHA1 */
    SDC_DGST_HASH_SHA224,              /**< DGST hash SHA224 */
    SDC_DGST_HASH_SHA256,              /**< DGST hash SHA256 */
    SDC_DGST_HASH_SHA384,              /**< DGST hash SHA384 */
    SDC_DGST_HASH_SHA512,              /**< DGST hash SHA512 */
    SDC_DGST_HASH_END                  /**< helper to iterate over hash modes */
} sdc_dgst_hash_t;

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Description of relevant parameters for a \ref sdc_wrap_unwrap_type_t
 */
typedef struct sdc_wrap_unwrap_desc sdc_wrap_unwrap_desc_t;

/**
 * \ingroup operations_encdec_common
 * \brief Description of relevant parameters for a \ref sdc_encrypt_decrypt_type_t
 */
typedef struct sdc_encrypt_decrypt_desc sdc_encrypt_decrypt_desc_t;

/**
 * \ingroup operations_signverify_common
 * \brief Description of relevant parameters for a \ref sdc_sign_verify_type_t
 */
typedef struct sdc_sign_verify_desc sdc_sign_verify_desc_t;

/**
 * \ingroup operations_digest_common
 * \brief Description of relevant parameters for a \ref sdc_dgst_type_t
 */
typedef struct sdc_dgst_desc sdc_dgst_desc_t;


/* Functions */

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Allocate \ref sdc_wrap_unwrap_desc_t
 *
 * \param[out] desc pointer to return the descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_desc_alloc(sdc_wrap_unwrap_desc_t **desc);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Free \ref sdc_wrap_unwrap_desc_t
 *
 * \param[in] desc descriptor to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_desc_free(sdc_wrap_unwrap_desc_t *desc);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief fill the \ref sdc_wrap_unwrap_desc_t descriptor with relevant parameters
 *
 * \param[in] session current session (key already loaded)
 * \param[in] type type selected for the operation
 * \param[out] desc descriptor to fill
 * \param[in] total_data_len length of the data to process
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_desc_fill (
    sdc_session_t *session,
    const sdc_wrap_unwrap_type_t *type,
    sdc_wrap_unwrap_desc_t *desc,
    size_t total_data_len);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get min, max, mod, default for tag length
 *
 * \param[in] desc pointer to descriptor
 * \param[out] min_val if != NULL used to return the min value
 * \param[out] max_val if != NULL used to return the max value
 * \param[out] mod if != NULL used to return the modulo of valid values between min and max
 *                 0 or 1 in case every value is valid
 * \param[out] default_val if != NULL used to return the default value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_wrap_unwrap_desc_get_tag(sdc_wrap_unwrap_desc_t *desc,
                                                 size_t *min_val,
                                                 size_t *max_val,
                                                 size_t *mod,
                                                 size_t *default_val);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get min, max, mod, default for tag length
 *
 * \param[in] desc pointer to descriptor
 * \param[out] min_val if != NULL used to return the min value
 * \param[out] max_val if != NULL used to return the max value
 * \param[out] mod if != NULL used to return the modulo of valid values between min and max
 *                 0 or 1 in case every value is valid
 * \param[out] default_val if != NULL used to return the default value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_wrap_unwrap_desc_get_iv(sdc_wrap_unwrap_desc_t *desc,
                                                size_t *min_val,
                                                size_t *max_val,
                                                size_t *mod,
                                                size_t *default_val);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get descriptor of the architecture dependent default algorithm/blockmode for wrap and unwrap
 *
 * \return pointer to constant memory block defining the default \ref sdc_wrap_unwrap_type_t descriptor of the current architecture
 */
SDC_API const sdc_wrap_unwrap_type_t *sdc_wrap_unwrap_get_default(void);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Allocate \ref sdc_wrap_unwrap_type_t
 *
 * The result is initialized with the architecture dependent default.
 * After usage it needs to be freed using \ref sdc_wrap_unwrap_type_free.\n
 * The content can be modified using \ref sdc_wrap_unwrap_type_set_algorithm,
 * \ref sdc_wrap_unwrap_type_set_block_mode and \ref sdc_wrap_unwrap_type_set_opt_bmsk.
 *
 * \param[out] type pointer to return the allocated type
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_alloc(sdc_wrap_unwrap_type_t **type);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Free \ref sdc_wrap_unwrap_type_t
 *
 * \param[in] type \ref sdc_wrap_unwrap_type_t type to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_free(sdc_wrap_unwrap_type_t *type);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Set algorithm of \ref sdc_wrap_unwrap_type_t
 *
 * Set the algorithm within \ref sdc_wrap_unwrap_type_t.
 * <b>This function needs to be called before \ref sdc_wrap_unwrap_type_set_block_mode</b>
 * When the algorithm is set to a valid value the block mode is changed
 * to the architecture dependent default block mode in combination with this
 * algorithm.
 *
 * \param[in] type \ref sdc_wrap_unwrap_type_t type
 * \param[in] alg \ref sdc_wrap_unwrap_alg_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or algorithm invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_set_algorithm(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t alg);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Set block mode of \ref sdc_wrap_unwrap_type_t
 *
 * Set the block mode within \ref sdc_wrap_unwrap_type_t.
 * <b>This function needs to be called after \ref sdc_wrap_unwrap_type_set_algorithm</b>
 * In case the block mode is not available in combination with the already
 * selected algorithm SDC_ALG_MODE_INVALID is returned.
 *
 * \param[in] type \ref sdc_wrap_unwrap_type_t type
 * \param[in] blk \ref sdc_wrap_unwrap_blk_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or block mode invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_set_block_mode(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t blk);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Set additional options of \ref sdc_wrap_unwrap_type_t (optional)
 *
 * Set the additional options for \ref sdc_wrap_unwrap_type_t.
 * <b>This function needs to be called after \ref sdc_wrap_unwrap_type_set_block_mode</b>
 * Note: Not all options will be supported for all algorithms, block modes,
 *       in all architectures or modes of operation
 * Note: Formatted API doesn't support any options (opt_bmsk != SDC_NO_OPT_BMSK)
 *
 * \param[in] type \ref sdc_wrap_unwrap_type_t type
 * \param[in] opt_bmsk
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return \ref SDC_OP_NOT_SUPPORTED - not supported option for the current alg + blockmode
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_set_opt_bmsk(sdc_wrap_unwrap_type_t *type, uint64_t opt_bmsk);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get algorithm of \ref sdc_wrap_unwrap_type_t
 *
 * \param[in] type \ref sdc_wrap_unwrap_type_t type
 * \param[out] alg pointer to return the algorithm
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_get_algorithm(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t *alg);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get block mode of \ref sdc_wrap_unwrap_type_t
 *
 * \param[in] type \ref sdc_wrap_unwrap_type_t type
 * \param[out] blk pointer to return the block mode
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - blk pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_get_block_mode(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t *blk);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get additional options of \ref sdc_wrap_unwrap_type_t
 *
 * \param[in] type \ref sdc_wrap_unwrap_type_t type
 * \param[out] opt_bmsk pointer to return the additional options
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - opt_bmsk pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_unwrap_type_get_opt_bmsk(const sdc_wrap_unwrap_type_t *type, uint64_t *opt_bmsk);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get name of algorithm

 * \param[in] alg \ref sdc_wrap_unwrap_alg_t algorithm
 *
 * \return name or NULL if invalid algorithm
 */
SDC_API const char* sdc_wrap_unwrap_algorithm_name (sdc_wrap_unwrap_alg_t alg);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get name of block mode

 * \param[in] blk \ref sdc_wrap_unwrap_blk_t block mode
 *
 * \return name or NULL if invalid block mode
 */
SDC_API const char* sdc_wrap_unwrap_block_mode_name (sdc_wrap_unwrap_blk_t blk);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get the key format and length (supported and default) by the corresponding type
 *
 * Note: in case returned fmt is a public key format the corresponding
 *       private key format can be used as well
 *
 * \param[in] type type selected
 * \param[out] sup_key_fmt_protect      if != NULL fill set supported key format - unprotect operation (sign, encrypt, wrap)
 * \param[out] sup_key_fmt_unprotect    if != NULL fill set supported key format - unprotect operation (verify, decrypt, unwrap)
 * \param[out] sup_key_lens if != NULL used to return the bmsk with all supported key lengths
 * \param[out] dflt_key_len  if != NULL used to return the default key length
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 */
SDC_API sdc_error_t sdc_wrap_unwrap_get_key_key_lens_fmt(
    sdc_wrap_unwrap_type_t *type,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len);

/**
 * \ingroup operations_encdec_common
 * \brief Allocate \ref sdc_encrypt_decrypt_desc_t
 *
 * \param[out] desc pointer to return the descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_desc_alloc(sdc_encrypt_decrypt_desc_t **desc);

/**
 * \ingroup operations_encdec_common
 * \brief Free \ref sdc_encrypt_decrypt_desc_t
 *
 * \param[in] desc descriptor to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_desc_free(sdc_encrypt_decrypt_desc_t *desc);

/**
 * \ingroup operations_encdec_common
 * \brief fill the \ref sdc_encrypt_decrypt_desc_t descriptor with relevant parameters
 *
 * \param[in] session current session (key already loaded)
 * \param[in] type type selected for the operation
 * \param[out] desc descriptor to fill
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_desc_fill (
    sdc_session_t *session,
    const sdc_encrypt_decrypt_type_t *type,
    sdc_encrypt_decrypt_desc_t *desc);

/**
 * \ingroup operations_encdec_common
 * \brief Get min, max, mod, default for IV
 *
 * \param[in] desc pointer to descriptor
 * \param[out] min_val if != NULL used to return the min value
 * \param[out] max_val if != NULL used to return the max value
 * \param[out] mod if != NULL used to return the modulo of valid values between min and max
 *                 0 or 1 in case every value is valid
 * \param[out] default_val if != NULL used to return the default value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_desc_get_iv(sdc_encrypt_decrypt_desc_t *desc,
                                                    size_t *min_val,
                                                    size_t *max_val,
                                                    size_t *mod,
                                                    size_t *default_val);

/**
 * \ingroup operations_encdec_common
 * \brief Get descriptor of the architecture dependent default algorithm/blockmode for encrypt and decrypt
 *
 * \return pointer to constant memory block defining the default \ref sdc_encrypt_decrypt_type_t descriptor of the current architecture
 */
SDC_API const sdc_encrypt_decrypt_type_t *sdc_encrypt_decrypt_get_default(void);

/**
 * \ingroup operations_encdec_common
 * \brief Allocate \ref sdc_encrypt_decrypt_type_t
 *
 * The result is initialized with the architecture dependent default.
 * After usage it needs to be freed using \ref sdc_encrypt_decrypt_type_free.\n
 * The content can be modified using \ref sdc_encrypt_decrypt_type_set_algorithm
 * \ref sdc_encrypt_decrypt_type_set_block_mode and \ref sdc_encrypt_decrypt_type_set_opt_bmsk.
 *
 * \param[out] type pointer to return the allocated type
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_alloc(sdc_encrypt_decrypt_type_t **type);

/**
 * \ingroup operations_encdec_common
 * \brief Free \ref sdc_encrypt_decrypt_type_t
 *
 * \param[in] type \ref sdc_encrypt_decrypt_type_t type to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_free(sdc_encrypt_decrypt_type_t *type);

/**
 * \ingroup operations_encdec_common
 * \brief Set algorithm of \ref sdc_encrypt_decrypt_type_t
 *
 * Set the algorithm within \ref sdc_encrypt_decrypt_type_t.
 * <b>This function needs to be called before \ref sdc_encrypt_decrypt_type_set_block_mode</b>
 * When the algorithm is set to a valid value the block mode is changed
 * to the architecture dependent default block mode in combination with this
 * algorithm.
 *
 * \param[in] type \ref sdc_encrypt_decrypt_type_t type
 * \param[in] alg \ref sdc_encrypt_decrypt_alg_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or algorithm invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_set_algorithm(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t alg);

/**
 * \ingroup operations_encdec_common
 * \brief Set block mode of \ref sdc_encrypt_decrypt_type_t
 *
 * Set the block mode within \ref sdc_encrypt_decrypt_type_t.
 * <b>This function needs to be called after \ref sdc_encrypt_decrypt_type_set_algorithm</b>
 * In case the block mode is not available in combination with the already
 * selected algorithm SDC_ALG_MODE_INVALID is returned.
 *
 * \param[in] type \ref sdc_encrypt_decrypt_type_t type
 * \param[in] blk \ref sdc_encrypt_decrypt_blk_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or block mode invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_set_block_mode(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t blk);

/**
 * \ingroup operations_encdec_common
 * \brief Set additional options of \ref sdc_encrypt_decrypt_type_t (optional)
 *
 * Set the additional options for \ref sdc_encrypt_decrypt_type_t.
 * <b>This function needs to be called after \ref sdc_encrypt_decrypt_type_set_block_mode</b>
 * Note: Not all options will be supported for all algorithms, block modes,
 *       in all architectures or modes of operation
 * Note: Formatted API doesn't support any options (opt_bmsk != SDC_NO_OPT_BMSK)
 *
 * \param[in] type \ref sdc_encrypt_decrypt_type_t type
 * \param[in] opt_bmsk
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return \ref SDC_OP_NOT_SUPPORTED - not supported option for the current alg + blockmode
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_set_opt_bmsk(sdc_encrypt_decrypt_type_t *type, uint64_t opt_bmsk);

/**
 * \ingroup operations_encdec_common
 * \brief Get algorithm of \ref sdc_encrypt_decrypt_type_t
 *
 * \param[in] type \ref sdc_encrypt_decrypt_type_t type
 * \param[out] alg pointer to return the algorithm
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_get_algorithm(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t *alg);

/**
 * \ingroup operations_encdec_common
 * \brief Get block mode of \ref sdc_encrypt_decrypt_type_t
 *
 * \param[in] type \ref sdc_encrypt_decrypt_type_t type
 * \param[out] blk pointer to return the block mode
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - blk pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_get_block_mode(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t *blk);

/**
 * \ingroup operations_encdec_common
 * \brief Get additional options of \ref sdc_encrypt_decrypt_type_t
 *
 * \param[in] type \ref sdc_encrypt_decrypt_type_t type
 * \param[out] opt_bmsk pointer to return the additional options
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - opt_bmsk pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_type_get_opt_bmsk(const sdc_encrypt_decrypt_type_t *type, uint64_t *opt_bmsk);

/**
 * \ingroup operations_encdec_common
 * \brief Get name of algorithm

 * \param[in] alg \ref sdc_encrypt_decrypt_alg_t algorithm
 *
 * \return name or NULL if invalid algorithm
 */
SDC_API const char* sdc_encrypt_decrypt_algorithm_name (sdc_encrypt_decrypt_alg_t alg);

/**
 * \ingroup operations_encdec_common
 * \brief Get name of block mode

 * \param[in] blk \ref sdc_encrypt_decrypt_blk_t block mode
 *
 * \return name or NULL if invalid block mode
 */
SDC_API const char* sdc_encrypt_decrypt_block_mode_name (sdc_encrypt_decrypt_blk_t blk);

/**
 * \ingroup operations_encdec_common
 * \brief Get the key format and length (supported and default) by the corresponding type
 *
 * Note: in case returned fmt is a public key format the corresponding
 *       private key format can be used as well
 *
 * \param[in] type type selected
 * \param[out] sup_key_fmt_protect      if != NULL fill set supported key format - unprotect operation (sign, encrypt, wrap)
 * \param[out] sup_key_fmt_unprotect    if != NULL fill set supported key format - unprotect operation (verify, decrypt, unwrap)
 * \param[out] sup_key_lens if != NULL used to return the bmsk with all supported key lengths
 * \param[out] dflt_key_len  if != NULL used to return the default key length
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_get_key_key_lens_fmt(
    sdc_encrypt_decrypt_type_t *type,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len);

/**
 * \ingroup operations_signverify_common
 * \brief Allocate \ref sdc_sign_verify_desc_t
 *
 * \param[out] desc pointer to return the descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_desc_alloc(sdc_sign_verify_desc_t **desc);

/**
 * \ingroup operations_signverify_common
 * \brief Free \ref sdc_sign_verify_desc_t
 *
 * \param[in] desc descriptor to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_desc_free(sdc_sign_verify_desc_t *desc);

/**
 * \ingroup operations_signverify_common
 * \brief fill the \ref sdc_sign_verify_desc_t descriptor with relevant parameters
 *
 * \param[in] session current session (key already loaded)
 * \param[in] type type selected for the operation
 * \param[out] desc descriptor to fill
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return \ref SDC_SESSION_INVALID - session is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_desc_fill (
    sdc_session_t *session,
    const sdc_sign_verify_type_t *type,
    sdc_sign_verify_desc_t *desc);

/**
 * \ingroup operations_signverify_common
 * \brief Get min, max, mod, default for tag length
 *
 * \param[in] desc pointer to descriptor
 * \param[out] min_val if != NULL used to return the min value
 * \param[out] max_val if != NULL used to return the max value
 * \param[out] mod if != NULL used to return the modulo of valid values between min and max
 *                 0 or 1 in case every value is valid
 * \param[out] default_val if != NULL used to return the default value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_sign_verify_desc_get_tag(sdc_sign_verify_desc_t *desc,
                                                 size_t *min_val,
                                                 size_t *max_val,
                                                 size_t *mod,
                                                 size_t *default_val);

/**
 * \ingroup operations_signverify_common
 * \brief Get min, max, mod, default for tag length
 *
 * \param[in] desc pointer to descriptor
 * \param[out] min_val if != NULL used to return the min value
 * \param[out] max_val if != NULL used to return the max value
 * \param[out] mod if != NULL used to return the modulo of valid values between min and max
 *                 0 or 1 in case every value is valid
 * \param[out] default_val if != NULL used to return the default value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_sign_verify_desc_get_iv(sdc_sign_verify_desc_t *desc,
                                                size_t *min_val,
                                                size_t *max_val,
                                                size_t *mod,
                                                size_t *default_val);

/**
 * \ingroup operations_signverify_common
 * \brief Get the key format and length (supported and default) by the corresponding type
 *
 * Note: in case returned fmt is a public key format the corresponding
 *       private key format can be used as well
 *
 * \param[in] type type selected
 * \param[out] sup_key_fmt_protect      if != NULL fill set supported key format - unprotect operation (sign, encrypt, wrap)
 * \param[out] sup_key_fmt_unprotect    if != NULL fill set supported key format - unprotect operation (verify, decrypt, unwrap)
 * \param[out] sup_key_lens if != NULL used to return the bmsk with all supported key lengths
 * \param[out] dflt_key_len  if != NULL used to return the default key length
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 */
SDC_API sdc_error_t sdc_sign_verify_get_key_key_lens_fmt(
    sdc_sign_verify_type_t *type,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len);

/**
 * \ingroup operations_signverify_common
 * \brief Get descriptor of the architecture dependent default algorithm/hash for sign and verify
 *
 * \return pointer to constant memory block defining the default \ref sdc_sign_verify_type_t descriptor of the current architecture
 */
SDC_API const sdc_sign_verify_type_t *sdc_sign_verify_get_default(void);

/**
 * \ingroup operations_signverify_common
 * \brief Allocate \ref sdc_sign_verify_type_t
 *
 * The result is initialized with the architecture dependent default.
 * After usage it needs to be freed using \ref sdc_sign_verify_type_free.\n
 * The content can be modified using \ref sdc_sign_verify_type_set_alg,
 * \ref sdc_sign_verify_type_set_hash and \ref sdc_sign_verify_type_set_opt_bmsk.
 *
 * \param[out] type pointer to return the allocated type
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_alloc(sdc_sign_verify_type_t **type);

/**
 * \ingroup operations_signverify_common
 * \brief Free \ref sdc_sign_verify_type_t
 *
 * \param[in] type \ref sdc_sign_verify_type_t type to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_free(sdc_sign_verify_type_t *type);

/**
 * \ingroup operations_signverify_common
 * \brief Set algorithm of \ref sdc_sign_verify_type_t
 *
 * Set the algorithm within \ref sdc_sign_verify_type_t.
 * <b>This function needs to be called before \ref sdc_sign_verify_type_set_hash</b>
 * When the algorithm is set to a valid value the hash is changed
 * to the architecture dependent default block mode in combination with this
 * algorithm.
 *
 * \param[in] type \ref sdc_sign_verify_type_t type
 * \param[in] alg \ref sdc_sign_verify_alg_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or alg invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_set_alg(sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t alg);

/**
 * \ingroup operations_signverify_common
 * \brief Set hash of \ref sdc_sign_verify_type_t
 *
 * Set the block mode within \ref sdc_sign_verify_type_t.
 * <b>This function needs to be called after \ref sdc_sign_verify_type_set_alg</b>
 * In case the hash is not available in combination with the already
 * selected algorithm SDC_ALG_MODE_INVALID is returned.
 *
 * \param[in] type \ref sdc_sign_verify_type_t type
 * \param[in] hash \ref sdc_sign_verify_hash_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or hash invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_set_hash(sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t hash);

/**
 * \ingroup operations_signverify_common
 * \brief Set additional options of \ref sdc_sign_verify_type_t (optional)
 *
 * Set the additional options for \ref sdc_sign_verify_type_t.
 * <b>This function needs to be called after \ref sdc_sign_verify_type_set_hash</b>
 * Note: Not all options will be supported for all algorithms, hashs,
 *       in all architectures or modes of operation
 * Note: Formatted API doesn't support any options (opt_bmsk != SDC_NO_OPT_BMSK)
 *
 * \param[in] type \ref sdc_sign_verify_type_t type
 * \param[in] opt_bmsk
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return \ref SDC_OP_NOT_SUPPORTED - not supported option for the current alg + hash
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_set_opt_bmsk(sdc_sign_verify_type_t *type, uint64_t opt_bmsk);

/**
 * \ingroup operations_signverify_common
 * \brief Get algorithm of \ref sdc_sign_verify_type_t
 *
 * \param[in] type \ref sdc_sign_verify_type_t type
 * \param[out] alg pointer to return the algorithm
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_get_alg(const sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t *alg);

/**
 * \ingroup operations_signverify_common
 * \brief Get hash of \ref sdc_sign_verify_type_t
 *
 * \param[in] type \ref sdc_sign_verify_type_t type
 * \param[out] hash pointer to return the hash
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - hash pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_get_hash(const sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t *hash);

/**
 * \ingroup operations_signverify_common
 * \brief Get additional options of \ref sdc_sign_verify_type_t
 *
 * \param[in] type \ref sdc_sign_verify_type_t type
 * \param[out] opt_bmsk pointer to return the additional options
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - opt_bmsk pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_verify_type_get_opt_bmsk(const sdc_sign_verify_type_t *type, uint64_t *opt_bmsk);

/**
 * \ingroup operations_signverify_common
 * \brief Get name of algorithm

 * \param[in] alg \ref sdc_sign_verify_alg_t alg
 *
 * \return name or NULL if invalid algorithm
 */
SDC_API const char* sdc_sign_verify_alg_name (sdc_sign_verify_alg_t alg);

/**
 * \ingroup operations_signverify_common
 * \brief Get name of hash

 * \param[in] hash \ref sdc_sign_verify_hash_t hash
 *
 * \return name or NULL if invalid hash
 */
SDC_API const char* sdc_sign_verify_hash_name (sdc_sign_verify_hash_t hash);

/**
 * \ingroup operations_digest_common
 * \brief Allocate \ref sdc_dgst_desc_t
 *
 * \param[out] desc pointer to return the descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_desc_alloc(sdc_dgst_desc_t **desc);

/**
 * \ingroup operations_digest_common
 * \brief Free \ref sdc_dgst_desc_t
 *
 * \param[in] desc descriptor to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_desc_free(sdc_dgst_desc_t *desc);

/**
 * \ingroup operations_digest_common
 * \brief fill the \ref sdc_dgst_desc_t descriptor with relevant parameters
 *
 * \param[in] type type selected for the operation
 * \param[out] desc descriptor to fill
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - desc invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_desc_fill (
    const sdc_dgst_type_t *type,
    sdc_dgst_desc_t *desc);

/**
 * \ingroup operations_digest_common
 * \brief Get min, max, mod, default for digest length
 *
 * \param[in] desc pointer to descriptor
 * \param[out] min_val if != NULL used to return the min value
 * \param[out] max_val if != NULL used to return the max value
 * \param[out] mod if != NULL used to return the modulo of valid values between min and max
 *                 0 or 1 in case every value is valid
 * \param[out] default_val if != NULL used to return the default value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_dgst_desc_get_digest(sdc_dgst_desc_t *desc,
                                             size_t *min_val,
                                             size_t *max_val,
                                             size_t *mod,
                                             size_t *default_val);

/**
 * \ingroup operations_digest_common
 * \brief Get descriptor of the architecture dependent default hash algorithm for digest
 *
 * \return pointer to constant memory block defining the default \ref sdc_dgst_type_t descriptor of the current architecture
 */
SDC_API const sdc_dgst_type_t *sdc_dgst_get_default(void);

/**
 * \ingroup operations_digest_common
 * \brief Allocate \ref sdc_dgst_type_t
 *
 * The result is initialized with the architecture dependent default.
 * After usage it needs to be freed using \ref sdc_dgst_type_free.\n
 * The content can be modified using \ref sdc_dgst_type_set_hash and
 * \ref sdc_dgst_type_set_opt_bmsk.
 *
 * \param[out] type pointer to return the allocated type
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_type_alloc(sdc_dgst_type_t **type);

/**
 * \ingroup operations_digest_common
 * \brief Free \ref sdc_dgst_type_t
 *
 * \param[in] type \ref sdc_dgst_type_t type to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_type_free(sdc_dgst_type_t *type);

/**
 * \ingroup operations_digest_common
 * \brief Set hash of \ref sdc_dgst_type_t
 *
 * Set the hash mode within \ref sdc_dgst_type_t.
 * In case the hash is not available SDC_ALG_MODE_INVALID is returned.
 *
 * \param[in] type \ref sdc_dgst_type_t type
 * \param[in] hash \ref sdc_dgst_hash_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or hash invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_type_set_hash(sdc_dgst_type_t *type, sdc_dgst_hash_t hash);

/**
 * \ingroup operations_digest_common
 * \brief Set additional options of \ref sdc_dgst_type_t (optional)
 *
 * Set the additional options for \ref sdc_dgst_type_t.
 * <b>This function needs to be called after \ref sdc_dgst_type_set_hash</b>
 * Note: Not all options will be supported for all algorithms, hashs,
 *       in all architectures or modes of operation
 *
 * \param[in] type \ref sdc_dgst_type_t type
 * \param[in] opt_bmsk
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return \ref SDC_OP_NOT_SUPPORTED - not supported option for the current alg + hash
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_type_set_opt_bmsk(sdc_dgst_type_t *type, uint64_t opt_bmsk);

/**
 * \ingroup operations_digest_common
 * \brief Get hash of \ref sdc_dgst_type_t
 *
 * \param[in] type \ref sdc_dgst_type_t type
 * \param[out] hash pointer to return the hash
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - hash pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_type_get_hash(const sdc_dgst_type_t *type, sdc_dgst_hash_t *hash);

/**
 * \ingroup operations_digest_common
 * \brief Get additional options of \ref sdc_dgst_type_t
 *
 * \param[in] type \ref sdc_dgst_type_t type
 * \param[out] opt_bmsk pointer to return the additional options
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - opt_bmsk pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_type_get_opt_bmsk(const sdc_dgst_type_t *type, uint64_t *opt_bmsk);

/**
 * \ingroup operations_digest_common
 * \brief Get name of hash

 * \param[in] hash \ref sdc_dgst_hash_t hash
 *
 * \return name or NULL if invalid hash
 */
SDC_API const char* sdc_dgst_hash_name (sdc_dgst_hash_t hash);



#ifdef __cplusplus
}
#endif

#endif
